<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\Admin\CreatePaginaRequest;
use App\Http\Requests\Admin\UpdatePaginaRequest;
use App\Models\Admin\Pagina;
use App\Repositories\Admin\PaginaRepository;
use App\Http\Controllers\AppBaseController;
use App\Repositories\StatuRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Flash;
use Prettus\Repository\Criteria\RequestCriteria;
use Response;

class PaginaController extends AppBaseController
{
    /** @var  PaginaRepository */
    private $paginaRepository;

    public function __construct(PaginaRepository $paginaRepo, StatuRepository $statuRepository)
    {
        $this->paginaRepository = $paginaRepo;
        $this->statuRepository = $statuRepository;
    }

    /**
     * Display a listing of the Pagina.
     *
     * @param Request $request
     * @return Response
     */
    public function index(Request $request)
    {
        $this->paginaRepository->pushCriteria(new RequestCriteria($request));
        $paginas = $this->paginaRepository->with('status')->all();

        return view('admin.paginas.index')->with('paginas', $paginas);
    }

    /**
     * Show the form for creating a new Pagina.
     *
     * @return Response
     */
    public function create()
    {
        $paginas = $this->paginaRepository->findWhere(['pagina_id' => null])->pluck('nome', 'id');
        $status = $this->statuRepository->statuSelectbox();
        return view('admin.paginas.create',compact('status', 'paginas'));
    }

    /**
     * Store a newly created Pagina in storage.
     *
     * @param CreatePaginaRequest $request
     *
     * @return Response
     */
    public function store(CreatePaginaRequest $request)
    {
        $input = $request->all();

        $input['slug'] = Str::slug($input['nome'], '-');


        $verifica = $this->paginaRepository->verificaSlug($input['slug']);

        $erro = '';

        $input['statu_id'] = 1;


        if(!$verifica){
            if (strlen(strip_tags($input['descricao'])) > 35) {
                if($input['pagina_id'] == 0){
                    $input['pagina_id'] = null;
                }
                $pagina = $this->paginaRepository->create_with_upload($input);

                Flash::success('Página salva com sucesso.');

                return redirect(route('admin.paginas.index'));
            } else {
                $erro = 'O campo conteúdo precisa ser preenchido com no mínimo 35 caracteres, tente novamente.';
            }
        } else{
            $erro = 'Já existe uma página com esse nome.';
        }

        Flash::error($erro);
        $status = $this->statuRepository->statuSelectbox();
        return view('admin.paginas.create',compact('status'))->with('pagina', $input);

    }


    /**
     * Show the form for editing the specified Pagina.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $pagina = $this->paginaRepository->findWithoutFail($id);

        if (empty($pagina)) {
            Flash::error('Pagina não encontrado(a)');

            return redirect(route('admin.paginas.index'));
        }
        $status = $this->statuRepository->statuSelectbox();
        $paginas = $this->paginaRepository->findWhere(['pagina_id' => null])->pluck('nome', 'id');


        return view('admin.paginas.edit',compact('status', 'paginas'))->with('pagina', $pagina);
    }

    /**
     * Update the specified Pagina in storage.
     *
     * @param  int              $id
     * @param UpdatePaginaRequest $request
     *
     * @return Response
     */
    public function update($id, UpdatePaginaRequest $request)
    {
        $pagina = $this->paginaRepository->findWithoutFail($id);

        if (empty($pagina)) {
            Flash::error('Pagina não encontrado(a)');

            return redirect(route('admin.paginas.index'));
        }

        $input = $request->all();

        $input['slug'] = Str::slug($input['nome'], '-');


        $erro = '';


        if (strlen(strip_tags($input['descricao'])) > 35) {
            if($input['pagina_id'] == 0){
                $input['pagina_id'] = null;
            }
            $pagina = $this->paginaRepository->update_with_upload($input, $id);
            Flash::success('Página editada com sucesso.');
            return redirect(route('admin.paginas.index'));

        } else {
            $erro = 'O campo conteúdo precisa ser preenchido com no mínimo 35 caracteres, tente novamente.';
        }


        Flash::error($erro);
        $status = $this->statuRepository->statuSelectbox();
        $pagina['nome'] = $input['nome'];
        $pagina['meta_description'] = $input['meta_description'];
        $pagina['meta_keyword'] = $input['meta_keyword'];
        $pagina['descricao'] = $input['descricao'];
        $pagina['ordem'] = $input['ordem'];
        return view('admin.paginas.edit',compact('status'))->with('pagina', $pagina);
    }

    /**
     * Remove the specified Pagina from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy($id)
    {
        $pagina = $this->paginaRepository->findWithoutFail($id);

        if (empty($pagina)) {
            Flash::error('Pagina não encontrado(a)');

            return redirect(route('admin.paginas.index'));
        }

        $this->paginaRepository->delete($id);

        Flash::success('Pagina deletado(a) com sucesso.');

        return redirect(route('admin.paginas.index'));
    }

    public function ativar($id)
    {
        $id = base64_decode($id);
        $this->paginaRepository->ativar($id);
        Flash::success('Pagina ativado(a) com sucesso.');
        return redirect(route('admin.paginas.index'));
    }

    public function inativar($id)
    {
        $id = base64_decode($id);
        $this->paginaRepository->inativar($id);
        Flash::success('Pagina inativado(a) com sucesso.');
        return redirect(route('admin.paginas.index'));
    }

}
